<?php
// Função para processar e formatar os MAC Addresses
function processarMacAddress($macBase64) {
    // Decodificar o Base64
    $mac = base64_decode($macBase64);

    // Remover caracteres não alfanuméricos e garantir 12 caracteres
    $mac = preg_replace('/[^A-Fa-f0-9]/', '', $mac);
    $mac = strtoupper(str_pad(substr($mac, 0, 12), 12, '0', STR_PAD_RIGHT));

    // Formatar como xx:xx:xx:xx:xx:xx
    return implode(':', str_split($mac, 2));
}

// Conectar ao banco de dados SQLite
$db = new SQLite3('permissoes.db');
$db->busyTimeout(5000); // Aguarda até 5 segundos antes de retornar "database is locked"

// Verificar se a requisição é POST
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Recebe os dados em formato JSON
    $data = json_decode(file_get_contents("php://input"), true);

    // Verifica se o parâmetro 'macAddresses' foi enviado
    if (isset($data['macAddresses']) && is_array($data['macAddresses'])) {
        $macAddresses = $data['macAddresses']; // Array de MAC Addresses base64
        $resultados = [];

        // Consulta para buscar mensagens da tabela permissoes_mensagem
        $mensagemStmt = $db->prepare("SELECT * FROM mensagens WHERE id = 1");
        $mensagemResult = $mensagemStmt->execute();
        $mensagemRow = $mensagemResult->fetchArray(SQLITE3_ASSOC);

        // Processar cada MAC Address
        foreach ($macAddresses as $macBase64) {
            $mac = processarMacAddress($macBase64);

            // Consultar o banco de dados com o MAC formatado
            $stmt = $db->prepare("SELECT permissao FROM permissoes WHERE mac_address = :mac");
            $stmt->bindValue(':mac', $mac, SQLITE3_TEXT);
            $result = $stmt->execute();

            $row = $result->fetchArray(SQLITE3_ASSOC);

            if ($row) {
                // Transformar as permissões em um array separando por vírgulas
                $permissoes = explode(',', $row['permissao']);

                $resultados[] = [
                    'mac' => $mac,
                    'permissao_filmes' => in_array('Filmes', $permissoes) ? 'sim' : 'não',
                    'permissao_canais' => in_array('Canais', $permissoes) ? 'sim' : 'não',
                    'permissao_series' => in_array('Séries', $permissoes) ? 'sim' : 'não'
                ];
            } else {
                // Caso o MAC não seja encontrado, todas as permissões estão ativas
                $resultados[] = [
                    'mac' => $mac,
                    'permissao_filmes' => 'sim',
                    'permissao_canais' => 'sim',
                    'permissao_series' => 'sim'
                ];
            }
        }

        // Adicionar as mensagens ao JSON de resposta
        $mensagens = [
            'mensagem_filmes' => $mensagemRow['filmes'] ?? '',
            'mensagem_series' => $mensagemRow['series'] ?? '',
            'mensagem_canais' => $mensagemRow['canais'] ?? ''
        ];

        // Retornar os resultados como JSON com suporte a acentos
        echo json_encode([
            'status' => 'sucesso',
            'dados' => $resultados,
            'mensagens' => $mensagens
        ], JSON_UNESCAPED_UNICODE);
    } else {
        echo json_encode([
            'status' => 'erro',
            'mensagem' => 'MAC Addresses inválidos ou não fornecidos.'
        ], JSON_UNESCAPED_UNICODE);
    }
} else {
    echo json_encode([
        'status' => 'erro',
        'mensagem' => 'Método de requisição inválido. Use POST.'
    ], JSON_UNESCAPED_UNICODE);
}

// Fecha a conexão com o banco
$db->close();
