<?php
/* =========================================================
   PORTALS FUNCTIONS – AUTO-DISABLE EMPTY URL
   ========================================================= */

define('PORTALS_FILE', dirname(__DIR__) . '/public/portals.json');

/* -------------------------
   WRITE MMM.JSON (RTXWave)
-------------------------- */
function write_mmm_json(array $portals): void {
    $baseDir = dirname(__DIR__);
    $out = [];

    // IMPORTANT:
    // Do NOT add placeholder (ID 0) to mmm.json
    // The app does not ignore it and shows empty DNS

    $id = 1;
    foreach ($portals as $p) {
        $enabled = isset($p['enabled']) ? (bool)$p['enabled'] : true;
        $name = trim((string)($p['name'] ?? ''));
        $url  = trim((string)($p['url']  ?? ''));

        if ($name === '' || $url === '') continue;
        if (!$enabled) continue;

        $out[] = [
            'id' => $id++,
            'name' => $name,
            'url' => $url,
            'epg_url' => '' // kept for compatibility
        ];
    }

    $json = json_encode(
        ['portals' => $out],
        JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES
    );

    // Root endpoint used by the app
    @file_put_contents($baseDir . '/mmm.json', $json);
}

/* -------------------------
   READ PORTALS
-------------------------- */
function read_portals(): array {
    if (!file_exists(PORTALS_FILE)) {
        return ['portals' => []];
    }

    $json = file_get_contents(PORTALS_FILE);
    $data = json_decode($json, true);

    if (!is_array($data) || !isset($data['portals'])) {
        return ['portals' => []];
    }

    return $data;
}

/* -------------------------
   SAVE PORTALS
   🔒 AUTO-DISABLE IF URL EMPTY
-------------------------- */
function save_portals(array $data): void {

    if (!isset($data['portals']) || !is_array($data['portals'])) {
        return;
    }

    foreach ($data['portals'] as &$p) {
        $p['name'] = trim($p['name'] ?? '');
        $p['url']  = trim($p['url'] ?? '');

        // 🚨 AUTO-DISABLE IF URL IS EMPTY
        if ($p['url'] === '') {
            $p['enabled'] = false;
        } else {
            $p['enabled'] = normalize_enabled($p['enabled'] ?? false);
        }
    }

    file_put_contents(
        PORTALS_FILE,
        json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)
    );
}

/* ---------------------------------------------------------
   APP / APK SAFE PORTALS
   ✅ enabled + non-empty only
---------------------------------------------------------- */
function get_enabled_portals(): array {

    $data = read_portals();
    $out = [];

    foreach ($data['portals'] as $p) {
        if (
            normalize_enabled($p['enabled'] ?? false) &&
            trim($p['name'] ?? '') !== '' &&
            trim($p['url'] ?? '') !== ''
        ) {
            $out[] = $p;
        }
    }

    return $out;
}

/* -------------------------
   NORMALIZE ENABLE FLAG
-------------------------- */
function normalize_enabled($value): bool {
    return (
        $value === true ||
        $value === 1 ||
        $value === '1' ||
        $value === 'true'
    );
}

/* -------------------------
   DASHBOARD COUNTER
-------------------------- */
function count_enabled_portals(): int {
    return count(get_enabled_portals());
}